/**
 * popup.js — ApplyOnce Popup Controller
 */
(function() {
  'use strict';
  
  const VAULT_KEY = 'applyonce_vault';
  const SALT_KEY = 'applyonce_salt';
  const FILLS_KEY = 'applyonce_fills';
  const FILLS_MONTH_KEY = 'applyonce_fills_month';
  const PRO_KEY = 'applyonce_pro';
  const FREE_LIMIT = 5;
  
  let unlockedProfile = null;
  
  const elStatusLocked = document.getElementById('status-locked');
  const elStatusReady = document.getElementById('status-ready');
  const elStatusEmpty = document.getElementById('status-empty');
  const elUnlockSection = document.getElementById('unlock-section');
  const elPassphrase = document.getElementById('passphrase');
  const elErrorMsg = document.getElementById('error-msg');
  const elBtnUnlock = document.getElementById('btn-unlock');
  const elBtnFill = document.getElementById('btn-fill');
  const elBtnPanic = document.getElementById('btn-panic');
  const elLinkOptions = document.getElementById('link-options');
  const elLinkSetup = document.getElementById('link-setup');
  const elLinkPrivacy = document.getElementById('link-privacy');
  
  // ─── Init ──────────────────────────────────────────────────────────────
  
  chrome.storage.local.get([VAULT_KEY, SALT_KEY], result => {
    if (!result[VAULT_KEY] || !result[SALT_KEY]) {
      // No profile stored
      showState('empty');
    } else {
      showState('locked');
    }
  });
  
  // ─── Event Handlers ────────────────────────────────────────────────────
  
  elBtnUnlock.addEventListener('click', async () => {
    const passphrase = elPassphrase.value;
    if (!passphrase) return;
    
    try {
      elBtnUnlock.disabled = true;
      elBtnUnlock.textContent = 'Unlocking...';
      
      unlockedProfile = await decryptVault(passphrase);
      showState('ready');
    } catch (err) {
      elErrorMsg.style.display = 'block';
      elPassphrase.value = '';
      elPassphrase.focus();
    } finally {
      elBtnUnlock.disabled = false;
      elBtnUnlock.textContent = 'Unlock';
    }
  });
  
  elPassphrase.addEventListener('keydown', e => {
    if (e.key === 'Enter') elBtnUnlock.click();
    elErrorMsg.style.display = 'none';
  });
  
  elBtnFill.addEventListener('click', async () => {
    if (!unlockedProfile) return;
    
    // Check free tier limit
    const isPro = await getProStatus();
    if (!isPro) {
      const fills = await getMonthlyFills();
      if (fills >= FREE_LIMIT) {
        elBtnFill.textContent = '🔒 Free limit reached';
        showUpgradePrompt();
        setTimeout(() => {
          elBtnFill.textContent = '📝 Fill This Form';
          elBtnFill.disabled = false;
        }, 3000);
        return;
      }
    }
    
    elBtnFill.disabled = true;
    elBtnFill.textContent = 'Filling...';
    
    try {
      const response = await chrome.runtime.sendMessage({
        action: 'trigger_fill',
        profile: unlockedProfile
      });
      
      if (response && response.ok) {
        elBtnFill.textContent = '✅ Review panel opened';
        elBtnPanic.style.display = 'block';
        // Track fill
        await incrementFills();
        updateFillCounter();
      } else {
        elBtnFill.textContent = '❌ Error — try refreshing page';
      }
    } catch (err) {
      elBtnFill.textContent = '❌ ' + (err.message || 'Error');
    }
    
    setTimeout(() => {
      elBtnFill.disabled = false;
      elBtnFill.textContent = '📝 Fill This Form';
    }, 3000);
  });
  
  elBtnPanic.addEventListener('click', async () => {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab) {
      chrome.tabs.sendMessage(tab.id, { action: 'panic' });
    }
    elBtnPanic.style.display = 'none';
  });
  
  elLinkOptions.addEventListener('click', e => {
    e.preventDefault();
    chrome.runtime.openOptionsPage();
  });
  
  if (elLinkSetup) {
    elLinkSetup.addEventListener('click', e => {
      e.preventDefault();
      chrome.runtime.openOptionsPage();
    });
  }
  
  if (elLinkPrivacy) {
    elLinkPrivacy.addEventListener('click', e => {
      e.preventDefault();
      chrome.tabs.create({ url: chrome.runtime.getURL('ui/options.html#privacy') });
    });
  }
  
  // ─── State Management ──────────────────────────────────────────────────
  
  function showState(state) {
    elStatusLocked.style.display = 'none';
    elStatusReady.style.display = 'none';
    elStatusEmpty.style.display = 'none';
    elUnlockSection.style.display = 'none';
    elBtnFill.disabled = true;
    
    switch (state) {
      case 'empty':
        elStatusEmpty.style.display = 'block';
        break;
      case 'locked':
        elStatusLocked.style.display = 'block';
        elUnlockSection.style.display = 'block';
        elPassphrase.focus();
        break;
      case 'ready':
        elStatusReady.style.display = 'block';
        elBtnFill.disabled = false;
        break;
    }
  }
  
  // ─── Crypto (inline — same as vault.js but for popup context) ─────────
  
  async function decryptVault(passphrase) {
    const result = await new Promise(resolve => {
      chrome.storage.local.get([VAULT_KEY, SALT_KEY], resolve);
    });
    
    const encrypted = result[VAULT_KEY];
    const saltB64 = result[SALT_KEY];
    if (!encrypted || !saltB64) throw new Error('No vault');
    
    const salt = base64ToArrayBuffer(saltB64);
    const enc = new TextEncoder();
    const keyMaterial = await crypto.subtle.importKey(
      'raw', enc.encode(passphrase), 'PBKDF2', false, ['deriveKey']
    );
    const key = await crypto.subtle.deriveKey(
      { name: 'PBKDF2', salt, iterations: 600000, hash: 'SHA-256' },
      keyMaterial,
      { name: 'AES-GCM', length: 256 },
      false,
      ['decrypt']
    );
    
    const iv = base64ToArrayBuffer(encrypted.iv);
    const ciphertext = base64ToArrayBuffer(encrypted.ciphertext);
    const decrypted = await crypto.subtle.decrypt({ name: 'AES-GCM', iv }, key, ciphertext);
    return JSON.parse(new TextDecoder().decode(decrypted));
  }
  
  function base64ToArrayBuffer(b64) {
    const bin = atob(b64);
    const bytes = new Uint8Array(bin.length);
    for (let i = 0; i < bin.length; i++) bytes[i] = bin.charCodeAt(i);
    return bytes;
  }
  
  // ─── Free Tier Tracking ────────────────────────────────────────────────
  
  function getCurrentMonth() {
    const d = new Date();
    return `${d.getFullYear()}-${String(d.getMonth()+1).padStart(2,'0')}`;
  }
  
  async function getMonthlyFills() {
    const result = await new Promise(r => chrome.storage.local.get([FILLS_KEY, FILLS_MONTH_KEY], r));
    if (result[FILLS_MONTH_KEY] !== getCurrentMonth()) return 0;
    return result[FILLS_KEY] || 0;
  }
  
  async function incrementFills() {
    const month = getCurrentMonth();
    const result = await new Promise(r => chrome.storage.local.get([FILLS_KEY, FILLS_MONTH_KEY], r));
    const count = (result[FILLS_MONTH_KEY] === month) ? (result[FILLS_KEY] || 0) + 1 : 1;
    await new Promise(r => chrome.storage.local.set({ [FILLS_KEY]: count, [FILLS_MONTH_KEY]: month }, r));
  }
  
  async function getProStatus() {
    const result = await new Promise(r => chrome.storage.local.get([PRO_KEY], r));
    return !!result[PRO_KEY];
  }
  
  async function updateFillCounter() {
    const isPro = await getProStatus();
    const counter = document.getElementById('fill-counter');
    if (!counter) return;
    if (isPro) {
      counter.textContent = '✨ Pro — Unlimited fills';
      counter.style.color = '#10b981';
    } else {
      const fills = await getMonthlyFills();
      const remaining = Math.max(0, FREE_LIMIT - fills);
      counter.textContent = `${remaining} free fills left this month`;
      counter.style.color = remaining <= 1 ? '#ef4444' : '#888';
    }
  }
  
  function showUpgradePrompt() {
    let modal = document.getElementById('upgrade-modal');
    if (modal) { modal.style.display = 'block'; return; }
    
    modal = document.createElement('div');
    modal.id = 'upgrade-modal';
    modal.style.cssText = 'position:fixed;top:0;left:0;width:100%;height:100%;background:rgba(0,0,0,0.85);display:flex;align-items:center;justify-content:center;z-index:9999;';
    modal.innerHTML = `
      <div style="background:#1a1a2e;border-radius:12px;padding:24px;width:280px;text-align:center;">
        <div style="font-size:32px;margin-bottom:8px;">⚡</div>
        <h3 style="color:#e5e5e5;margin-bottom:8px;">You've hit the free limit</h3>
        <p style="color:#888;font-size:13px;margin-bottom:16px;">Upgrade to Pro for unlimited fills.<br>$6.99/mo or $49.99/year.</p>
        <button id="btn-upgrade" style="display:block;width:100%;padding:12px;border:none;border-radius:8px;background:#10b981;color:white;font-size:15px;font-weight:600;cursor:pointer;margin-bottom:8px;">Upgrade to Pro</button>
        <button id="btn-dismiss" style="display:block;width:100%;padding:8px;border:none;background:transparent;color:#666;font-size:13px;cursor:pointer;">Maybe later</button>
      </div>
    `;
    document.body.appendChild(modal);
    
    document.getElementById('btn-upgrade').addEventListener('click', () => {
      chrome.tabs.create({ url: 'https://buy.stripe.com/applyonce_pro' }); // Replace with real Stripe link
    });
    document.getElementById('btn-dismiss').addEventListener('click', () => {
      modal.style.display = 'none';
    });
  }
  
  // Init fill counter on load
  updateFillCounter();
  
})();
