/**
 * options.js — Profile Editor Controller
 */
(function() {
  'use strict';
  
  const VAULT_KEY = 'applyonce_vault';
  const SALT_KEY = 'applyonce_salt';
  const ITERATIONS = 600000;
  
  let addressCount = 0;
  let educationCount = 0;
  
  // ─── Address/Education Templates ───────────────────────────────────────
  
  function addAddressEntry(data = {}) {
    const idx = addressCount++;
    const container = document.getElementById('addresses-container');
    const div = document.createElement('div');
    div.className = 'address-entry';
    div.dataset.idx = idx;
    div.innerHTML = `
      <button class="remove-btn" data-remove-address="${idx}">✕</button>
      <div class="form-group"><label>Label</label><input data-addr="${idx}" data-field="label" value="${esc(data.label || (idx === 0 ? 'Current' : ''))}"></div>
      <div class="form-group"><label>Street Address</label><input data-addr="${idx}" data-field="street" value="${esc(data.street || '')}"></div>
      <div class="form-group"><label>Apt/Suite</label><input data-addr="${idx}" data-field="street2" value="${esc(data.street2 || '')}"></div>
      <div class="row">
        <div class="form-group"><label>City</label><input data-addr="${idx}" data-field="city" value="${esc(data.city || '')}"></div>
        <div class="form-group"><label>State</label><input data-addr="${idx}" data-field="state" value="${esc(data.state || '')}"></div>
        <div class="form-group"><label>ZIP</label><input data-addr="${idx}" data-field="zip" value="${esc(data.zip || '')}"></div>
      </div>
      <div class="row">
        <div class="form-group"><label>Country</label><input data-addr="${idx}" data-field="country" value="${esc(data.country || 'US')}"></div>
        <div class="form-group"><label>Start Date</label><input type="date" data-addr="${idx}" data-field="startDate" value="${esc(data.startDate || '')}"></div>
        <div class="form-group"><label>End Date</label><input type="date" data-addr="${idx}" data-field="endDate" value="${esc(data.endDate || '')}"></div>
      </div>
    `;
    container.appendChild(div);
    div.querySelector(`[data-remove-address="${idx}"]`).addEventListener('click', () => div.remove());
  }
  
  function addEducationEntry(data = {}) {
    const idx = educationCount++;
    const container = document.getElementById('education-container');
    const div = document.createElement('div');
    div.className = 'education-entry';
    div.dataset.idx = idx;
    div.innerHTML = `
      <button class="remove-btn" data-remove-edu="${idx}">✕</button>
      <div class="form-group"><label>School Name</label><input data-edu="${idx}" data-field="school" value="${esc(data.school || '')}"></div>
      <div class="row">
        <div class="form-group"><label>Degree</label><input data-edu="${idx}" data-field="degree" value="${esc(data.degree || '')}"></div>
        <div class="form-group"><label>Major</label><input data-edu="${idx}" data-field="major" value="${esc(data.major || '')}"></div>
        <div class="form-group"><label>GPA</label><input data-edu="${idx}" data-field="gpa" value="${esc(data.gpa || '')}"></div>
      </div>
      <div class="row">
        <div class="form-group"><label>Start Date</label><input type="date" data-edu="${idx}" data-field="startDate" value="${esc(data.startDate || '')}"></div>
        <div class="form-group"><label>End Date</label><input type="date" data-edu="${idx}" data-field="endDate" value="${esc(data.endDate || '')}"></div>
        <div class="form-group"><label>Graduation</label><input type="date" data-edu="${idx}" data-field="graduationDate" value="${esc(data.graduationDate || '')}"></div>
      </div>
    `;
    container.appendChild(div);
    div.querySelector(`[data-remove-edu="${idx}"]`).addEventListener('click', () => div.remove());
  }
  
  // ─── Collect Form Data ─────────────────────────────────────────────────
  
  function collectProfile() {
    const profile = {
      identity: {
        firstName: val('firstName'),
        middleName: val('middleName'),
        lastName: val('lastName'),
        preferredName: val('preferredName'),
        email: val('email'),
        phone: val('phone'),
        dateOfBirth: val('dateOfBirth'),
        gender: val('gender'),
        ssn: ''
      },
      addresses: [],
      education: [],
      parent: {
        name: val('parentName'),
        email: val('parentEmail'),
        phone: val('parentPhone'),
        relationship: val('parentRelationship')
      }
    };
    
    // Collect addresses
    const addrEntries = document.querySelectorAll('.address-entry');
    addrEntries.forEach(entry => {
      const idx = entry.dataset.idx;
      const fields = ['label', 'street', 'street2', 'city', 'state', 'zip', 'country', 'startDate', 'endDate'];
      const addr = {};
      fields.forEach(f => { addr[f] = (entry.querySelector(`[data-addr="${idx}"][data-field="${f}"]`) || {}).value || ''; });
      profile.addresses.push(addr);
    });
    
    // Collect education
    const eduEntries = document.querySelectorAll('.education-entry');
    eduEntries.forEach(entry => {
      const idx = entry.dataset.idx;
      const fields = ['school', 'degree', 'major', 'gpa', 'startDate', 'endDate', 'graduationDate'];
      const edu = {};
      fields.forEach(f => { edu[f] = (entry.querySelector(`[data-edu="${idx}"][data-field="${f}"]`) || {}).value || ''; });
      profile.education.push(edu);
    });
    
    if (profile.addresses.length === 0) profile.addresses.push({ label: 'Current', street: '', street2: '', city: '', state: '', zip: '', country: 'US', startDate: '', endDate: '' });
    if (profile.education.length === 0) profile.education.push({ school: '', degree: '', major: '', gpa: '', startDate: '', endDate: '', graduationDate: '' });
    
    return profile;
  }
  
  function populateForm(profile) {
    setVal('firstName', profile.identity?.firstName);
    setVal('middleName', profile.identity?.middleName);
    setVal('lastName', profile.identity?.lastName);
    setVal('preferredName', profile.identity?.preferredName);
    setVal('email', profile.identity?.email);
    setVal('phone', profile.identity?.phone);
    setVal('dateOfBirth', profile.identity?.dateOfBirth);
    setVal('gender', profile.identity?.gender);
    setVal('parentName', profile.parent?.name);
    setVal('parentEmail', profile.parent?.email);
    setVal('parentPhone', profile.parent?.phone);
    setVal('parentRelationship', profile.parent?.relationship);
    
    document.getElementById('addresses-container').innerHTML = '';
    addressCount = 0;
    (profile.addresses || [{}]).forEach(a => addAddressEntry(a));
    
    document.getElementById('education-container').innerHTML = '';
    educationCount = 0;
    (profile.education || [{}]).forEach(e => addEducationEntry(e));
  }
  
  // ─── Crypto ────────────────────────────────────────────────────────────
  
  async function encryptAndSave(profile, passphrase) {
    let saltBytes;
    const stored = await storageGet(SALT_KEY);
    if (stored) {
      saltBytes = b64ToArr(stored);
    } else {
      saltBytes = crypto.getRandomValues(new Uint8Array(16));
      await storageSet(SALT_KEY, arrToB64(saltBytes));
    }
    
    const key = await deriveKey(passphrase, saltBytes);
    const iv = crypto.getRandomValues(new Uint8Array(12));
    const ct = await crypto.subtle.encrypt({ name: 'AES-GCM', iv }, key, new TextEncoder().encode(JSON.stringify(profile)));
    
    await storageSet(VAULT_KEY, { iv: arrToB64(iv), ciphertext: arrToB64(ct) });
  }
  
  async function decryptAndLoad(passphrase) {
    const encrypted = await storageGet(VAULT_KEY);
    const saltB64 = await storageGet(SALT_KEY);
    if (!encrypted || !saltB64) return null;
    
    const salt = b64ToArr(saltB64);
    const key = await deriveKey(passphrase, salt);
    const decrypted = await crypto.subtle.decrypt(
      { name: 'AES-GCM', iv: b64ToArr(encrypted.iv) }, key, b64ToArr(encrypted.ciphertext)
    );
    return JSON.parse(new TextDecoder().decode(decrypted));
  }
  
  async function deriveKey(passphrase, salt) {
    const km = await crypto.subtle.importKey('raw', new TextEncoder().encode(passphrase), 'PBKDF2', false, ['deriveKey']);
    return crypto.subtle.deriveKey(
      { name: 'PBKDF2', salt, iterations: ITERATIONS, hash: 'SHA-256' },
      km, { name: 'AES-GCM', length: 256 }, false, ['encrypt', 'decrypt']
    );
  }
  
  // ─── Event Handlers ────────────────────────────────────────────────────
  
  document.getElementById('btn-save').addEventListener('click', async () => {
    const passphrase = document.getElementById('passphrase').value;
    if (!passphrase) { toast('Enter a passphrase first', 'error'); return; }
    if (passphrase.length < 4) { toast('Passphrase too short', 'error'); return; }
    
    const profile = collectProfile();
    try {
      await encryptAndSave(profile, passphrase);
      toast('✅ Profile saved & encrypted');
    } catch (err) {
      toast('❌ Error saving: ' + err.message, 'error');
    }
  });
  
  document.getElementById('btn-load').addEventListener('click', async () => {
    const passphrase = document.getElementById('passphrase').value;
    if (!passphrase) { toast('Enter your passphrase', 'error'); return; }
    
    try {
      const profile = await decryptAndLoad(passphrase);
      if (!profile) { toast('No saved profile found', 'error'); return; }
      populateForm(profile);
      toast('✅ Profile loaded');
    } catch (err) {
      toast('❌ Wrong passphrase', 'error');
    }
  });
  
  document.getElementById('btn-export').addEventListener('click', async () => {
    const encrypted = await storageGet(VAULT_KEY);
    const salt = await storageGet(SALT_KEY);
    if (!encrypted || !salt) { toast('Nothing to export', 'error'); return; }
    
    const data = JSON.stringify({ version: 1, salt, encrypted }, null, 2);
    const blob = new Blob([data], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url; a.download = 'applyonce-backup.json'; a.click();
    URL.revokeObjectURL(url);
    toast('📤 Backup exported');
  });
  
  document.getElementById('btn-import').addEventListener('click', () => {
    document.getElementById('import-file').click();
  });
  
  document.getElementById('import-file').addEventListener('change', async (e) => {
    const file = e.target.files[0];
    if (!file) return;
    try {
      const text = await file.text();
      const data = JSON.parse(text);
      if (data.version !== 1) throw new Error('Invalid backup');
      await storageSet(SALT_KEY, data.salt);
      await storageSet(VAULT_KEY, data.encrypted);
      toast('📥 Backup imported. Click Load to decrypt.');
    } catch (err) {
      toast('❌ Invalid backup file', 'error');
    }
  });
  
  document.getElementById('btn-clear').addEventListener('click', async () => {
    if (!confirm('Delete all stored profile data? This cannot be undone.')) return;
    await new Promise(r => chrome.storage.local.remove([VAULT_KEY, SALT_KEY], r));
    populateForm({ identity: {}, addresses: [{}], education: [{}], parent: {} });
    document.getElementById('passphrase').value = '';
    toast('🗑️ All data cleared');
  });
  
  document.getElementById('add-address').addEventListener('click', () => addAddressEntry());
  document.getElementById('add-education').addEventListener('click', () => addEducationEntry());
  
  // ─── Helpers ───────────────────────────────────────────────────────────
  
  function val(id) { return (document.getElementById(id) || {}).value || ''; }
  function setVal(id, v) { const el = document.getElementById(id); if (el) el.value = v || ''; }
  function esc(s) { return (s || '').replace(/"/g, '&quot;').replace(/</g, '&lt;'); }
  
  function toast(msg, type = 'success') {
    const el = document.getElementById('toast');
    el.textContent = msg;
    el.style.background = type === 'error' ? '#dc2626' : '#10b981';
    el.style.display = 'block';
    setTimeout(() => { el.style.display = 'none'; }, 3000);
  }
  
  function storageGet(key) {
    return new Promise(r => chrome.storage.local.get(key, res => r(res[key])));
  }
  function storageSet(key, value) {
    return new Promise(r => chrome.storage.local.set({ [key]: value }, r));
  }
  function arrToB64(buf) {
    const b = buf instanceof Uint8Array ? buf : new Uint8Array(buf);
    let s = ''; for (let i = 0; i < b.length; i++) s += String.fromCharCode(b[i]);
    return btoa(s);
  }
  function b64ToArr(b64) {
    const s = atob(b64); const b = new Uint8Array(s.length);
    for (let i = 0; i < s.length; i++) b[i] = s.charCodeAt(i);
    return b;
  }
  
  // ─── Init ──────────────────────────────────────────────────────────────
  addAddressEntry();
  addEducationEntry();
  
  // Scroll to privacy section if hash
  if (location.hash === '#privacy') {
    setTimeout(() => document.getElementById('privacy').scrollIntoView({ behavior: 'smooth' }), 100);
  }
  
})();
