/**
 * service_worker.js — ApplyOnce MV3 Background
 * 
 * Handles messages between popup and content scripts.
 * Manages site allowlist.
 * Injects content script + synonyms when user triggers fill.
 */

// Show welcome page on first install
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    chrome.tabs.create({ url: chrome.runtime.getURL('ui/welcome.html') });
  }
});

// Load synonyms at startup
let synonymsData = {};
fetch(chrome.runtime.getURL('core/synonyms.json'))
  .then(r => r.json())
  .then(data => { synonymsData = data; })
  .catch(err => console.error('Failed to load synonyms:', err));

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  
  if (msg.action === 'trigger_fill') {
    // User clicked "Fill This Form" in popup
    handleFillTrigger(msg.profile)
      .then(result => sendResponse(result))
      .catch(err => sendResponse({ ok: false, error: err.message }));
    return true; // async response
  }
  
  if (msg.action === 'check_allowlist') {
    checkAllowlist(msg.domain)
      .then(allowed => sendResponse({ allowed }))
      .catch(() => sendResponse({ allowed: false }));
    return true;
  }
  
  if (msg.action === 'add_to_allowlist') {
    addToAllowlist(msg.domain)
      .then(() => sendResponse({ ok: true }))
      .catch(err => sendResponse({ ok: false, error: err.message }));
    return true;
  }
  
  if (msg.action === 'get_allowlist') {
    getAllowlist()
      .then(list => sendResponse({ list }))
      .catch(() => sendResponse({ list: [] }));
    return true;
  }
  
  if (msg.action === 'remove_from_allowlist') {
    removeFromAllowlist(msg.domain)
      .then(() => sendResponse({ ok: true }))
      .catch(err => sendResponse({ ok: false, error: err.message }));
    return true;
  }
});

/**
 * Inject content script with synonyms and trigger fill flow.
 */
async function handleFillTrigger(profile) {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab) throw new Error('No active tab');
  
  // First inject the synonyms as a global variable (all frames for iframe support)
  await chrome.scripting.executeScript({
    target: { tabId: tab.id, allFrames: true },
    func: (syns) => { window.__applyonce_synonyms = syns; },
    args: [synonymsData]
  });
  
  // Then inject the content script into all frames (Greenhouse uses iframes)
  await chrome.scripting.executeScript({
    target: { tabId: tab.id, allFrames: true },
    files: ['content_script.js']
  });
  
  // Send the profile to the content script to start filling
  const response = await chrome.tabs.sendMessage(tab.id, {
    action: 'fill_form',
    profile: profile
  });
  
  return response;
}

// ─── Site Allowlist ────────────────────────────────────────────────────────

async function getAllowlist() {
  return new Promise(resolve => {
    chrome.storage.local.get('applyonce_allowlist', result => {
      resolve(result.applyonce_allowlist || []);
    });
  });
}

async function checkAllowlist(domain) {
  const list = await getAllowlist();
  return list.includes(domain);
}

async function addToAllowlist(domain) {
  const list = await getAllowlist();
  if (!list.includes(domain)) {
    list.push(domain);
    await new Promise(resolve => {
      chrome.storage.local.set({ applyonce_allowlist: list }, resolve);
    });
  }
}

async function removeFromAllowlist(domain) {
  let list = await getAllowlist();
  list = list.filter(d => d !== domain);
  await new Promise(resolve => {
    chrome.storage.local.set({ applyonce_allowlist: list }, resolve);
  });
}
