/**
 * vault.js — Encrypted Profile Storage
 * 
 * AES-GCM encryption with PBKDF2 key derivation.
 * All profile data stored encrypted in chrome.storage.local.
 * Never transmits user data anywhere.
 */

const VAULT_STORAGE_KEY = 'applyonce_vault';
const VAULT_SALT_KEY = 'applyonce_salt';
const PBKDF2_ITERATIONS = 600000;

/**
 * Derive an AES-GCM key from a passphrase using PBKDF2.
 */
async function deriveKey(passphrase, salt) {
  const enc = new TextEncoder();
  const keyMaterial = await crypto.subtle.importKey(
    'raw', enc.encode(passphrase), 'PBKDF2', false, ['deriveKey']
  );
  return crypto.subtle.deriveKey(
    { name: 'PBKDF2', salt, iterations: PBKDF2_ITERATIONS, hash: 'SHA-256' },
    keyMaterial,
    { name: 'AES-GCM', length: 256 },
    false,
    ['encrypt', 'decrypt']
  );
}

/**
 * Encrypt a profile object.
 * Returns { iv, ciphertext } as base64 strings.
 */
async function encryptProfile(profile, passphrase, salt) {
  const key = await deriveKey(passphrase, salt);
  const iv = crypto.getRandomValues(new Uint8Array(12));
  const enc = new TextEncoder();
  const ciphertext = await crypto.subtle.encrypt(
    { name: 'AES-GCM', iv },
    key,
    enc.encode(JSON.stringify(profile))
  );
  return {
    iv: arrayBufferToBase64(iv),
    ciphertext: arrayBufferToBase64(ciphertext)
  };
}

/**
 * Decrypt a profile object.
 * Returns the parsed profile or throws on wrong passphrase.
 */
async function decryptProfile(encrypted, passphrase, salt) {
  const key = await deriveKey(passphrase, salt);
  const iv = base64ToArrayBuffer(encrypted.iv);
  const ciphertext = base64ToArrayBuffer(encrypted.ciphertext);
  const decrypted = await crypto.subtle.decrypt(
    { name: 'AES-GCM', iv },
    key,
    ciphertext
  );
  const dec = new TextDecoder();
  return JSON.parse(dec.decode(decrypted));
}

/**
 * Save profile to chrome.storage.local (encrypted).
 */
async function saveProfile(profile, passphrase) {
  let saltBytes;
  const stored = await storageGet(VAULT_SALT_KEY);
  if (stored) {
    saltBytes = base64ToArrayBuffer(stored);
  } else {
    saltBytes = crypto.getRandomValues(new Uint8Array(16));
    await storageSet(VAULT_SALT_KEY, arrayBufferToBase64(saltBytes));
  }
  const encrypted = await encryptProfile(profile, passphrase, saltBytes);
  await storageSet(VAULT_STORAGE_KEY, encrypted);
  return true;
}

/**
 * Load profile from chrome.storage.local (decrypted).
 */
async function loadProfile(passphrase) {
  const encrypted = await storageGet(VAULT_STORAGE_KEY);
  const saltB64 = await storageGet(VAULT_SALT_KEY);
  if (!encrypted || !saltB64) return null;
  const salt = base64ToArrayBuffer(saltB64);
  return decryptProfile(encrypted, passphrase, salt);
}

/**
 * Export encrypted vault as a JSON file (for backup).
 */
async function exportVault() {
  const encrypted = await storageGet(VAULT_STORAGE_KEY);
  const salt = await storageGet(VAULT_SALT_KEY);
  if (!encrypted || !salt) return null;
  return JSON.stringify({ version: 1, salt, encrypted }, null, 2);
}

/**
 * Import encrypted vault from a JSON backup string.
 */
async function importVault(jsonString) {
  const data = JSON.parse(jsonString);
  if (data.version !== 1 || !data.salt || !data.encrypted) {
    throw new Error('Invalid vault backup format');
  }
  await storageSet(VAULT_SALT_KEY, data.salt);
  await storageSet(VAULT_STORAGE_KEY, data.encrypted);
  return true;
}

/**
 * Get a default empty profile structure.
 */
function getEmptyProfile() {
  return {
    identity: {
      firstName: '',
      middleName: '',
      lastName: '',
      preferredName: '',
      email: '',
      phone: '',
      dateOfBirth: '',
      gender: '',
      ethnicity: '',
      ssn: '',
      citizenship: '',
      workAuthorization: '',
      veteranStatus: '',
      disabilityStatus: '',
      linkedIn: '',
      website: ''
    },
    addresses: [{
      label: 'Current',
      street: '',
      street2: '',
      city: '',
      state: '',
      zip: '',
      country: 'US',
      startDate: '',
      endDate: ''
    }],
    education: [{
      school: '',
      degree: '',
      major: '',
      minor: '',
      gpa: '',
      startDate: '',
      endDate: '',
      graduationDate: '',
      honors: ''
    }],
    work: [{
      company: '',
      title: '',
      startDate: '',
      endDate: '',
      current: false,
      description: ''
    }],
    parent: {
      name: '',
      email: '',
      phone: '',
      relationship: ''
    }
  };
}

// ─── Storage helpers (work in extension and test environments) ──────────────

function storageGet(key) {
  if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
    return new Promise(resolve => {
      chrome.storage.local.get(key, result => resolve(result[key]));
    });
  }
  // Fallback for testing
  if (typeof globalThis.__mockStorage !== 'undefined') {
    return Promise.resolve(globalThis.__mockStorage[key]);
  }
  return Promise.resolve(undefined);
}

function storageSet(key, value) {
  if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
    return new Promise(resolve => {
      chrome.storage.local.set({ [key]: value }, resolve);
    });
  }
  if (typeof globalThis.__mockStorage !== 'undefined') {
    globalThis.__mockStorage[key] = value;
    return Promise.resolve();
  }
  return Promise.resolve();
}

// ─── Encoding helpers ──────────────────────────────────────────────────────

function arrayBufferToBase64(buffer) {
  const bytes = buffer instanceof Uint8Array ? buffer : new Uint8Array(buffer);
  let binary = '';
  for (let i = 0; i < bytes.byteLength; i++) {
    binary += String.fromCharCode(bytes[i]);
  }
  return btoa(binary);
}

function base64ToArrayBuffer(base64) {
  const binary = atob(base64);
  const bytes = new Uint8Array(binary.length);
  for (let i = 0; i < binary.length; i++) {
    bytes[i] = binary.charCodeAt(i);
  }
  return bytes;
}

// ─── Exports ───────────────────────────────────────────────────────────────

if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    encryptProfile, decryptProfile, deriveKey,
    saveProfile, loadProfile,
    exportVault, importVault,
    getEmptyProfile,
    arrayBufferToBase64, base64ToArrayBuffer,
    PBKDF2_ITERATIONS
  };
}
