/**
 * signatures.js — Field Signature Builder
 * 
 * Detects fillable form fields on a page and builds a "signature" for each
 * containing label text, name/id, type, and surrounding context.
 * This signature is used by mapper.js to match profile fields.
 */

/**
 * Build field signatures for all fillable elements in a document/container.
 * @param {Document|Element} root - The document or container to scan.
 * @returns {Array<{element: Element, signature: Object}>}
 */
function buildFieldSignatures(root) {
  const doc = root.ownerDocument || root;
  const selectors = [
    'input[type="text"]',
    'input[type="email"]',
    'input[type="tel"]',
    'input[type="date"]',
    'input[type="number"]',
    'input[type="url"]',
    'input[type="radio"]',
    'input[type="checkbox"]',
    'input:not([type])',  // Default type is text
    'textarea',
    'select'
  ];
  
  // Collect elements from regular DOM
  const elements = Array.from(root.querySelectorAll(selectors.join(',')));
  
  // Also traverse Shadow DOM roots (for Workday Web Components)
  const shadowElements = collectShadowDOMFields(root, selectors);
  elements.push(...shadowElements);
  
  const results = [];
  
  for (const el of elements) {
    // Skip hidden, disabled, or submit-related fields
    if (el.type === 'hidden' || el.type === 'submit' || el.type === 'button') continue;
    if (el.disabled) continue;
    
    const sig = {
      name: (el.name || '').toLowerCase().trim(),
      id: (el.id || '').toLowerCase().trim(),
      type: (el.type || 'text').toLowerCase(),
      tagName: el.tagName.toLowerCase(),
      placeholder: (el.placeholder || '').toLowerCase().trim(),
      ariaLabel: (el.getAttribute('aria-label') || '').toLowerCase().trim(),
      labelText: extractLabelText(el, doc).toLowerCase().trim(),
      autocomplete: (el.getAttribute('autocomplete') || '').toLowerCase().trim(),
      dataAutomationId: (el.getAttribute('data-automation-id') || '').toLowerCase().trim(),
      sectionHeading: findSectionHeading(el).toLowerCase().trim(),
      // Combined text for fuzzy matching
      allText: ''
    };
    
    // Build combined text from all signals (including data-automation-id for Workday)
    sig.allText = [
      sig.name, sig.id, sig.placeholder,
      sig.ariaLabel, sig.labelText, sig.autocomplete,
      sig.dataAutomationId
    ].filter(Boolean).join(' ').toLowerCase();
    
    results.push({ element: el, signature: sig });
  }
  
  return results;
}

/**
 * Extract the label text for a form element.
 * Checks: explicit <label for="...">, parent <label>, aria-labelledby, closest text.
 */
function extractLabelText(el, doc) {
  // 1. Explicit label via for= attribute
  if (el.id) {
    const escapedId = typeof CSS !== 'undefined' && CSS.escape ? CSS.escape(el.id) : el.id.replace(/([^\w-])/g, '\\$1');
    const label = doc.querySelector(`label[for="${escapedId}"]`);
    if (label) return label.textContent.trim();
  }
  
  // 2. Wrapping <label> parent
  const parentLabel = el.closest('label');
  if (parentLabel) {
    // Get text content excluding the input's own text
    const clone = parentLabel.cloneNode(true);
    const inputs = clone.querySelectorAll('input,select,textarea');
    inputs.forEach(inp => inp.remove());
    const text = clone.textContent.trim();
    if (text) return text;
  }
  
  // 3. aria-labelledby
  const labelledBy = el.getAttribute('aria-labelledby');
  if (labelledBy) {
    const labelEl = doc.getElementById(labelledBy);
    if (labelEl) return labelEl.textContent.trim();
  }
  
  // 4. Parent td/th — check previous td/th in same row
  const td = el.closest('td');
  if (td) {
    const prevTd = td.previousElementSibling;
    if (prevTd && prevTd.textContent.trim().length < 100) {
      return prevTd.textContent.trim();
    }
  }
  
  // 5. Previous sibling text — but NOT table/div containers that may hold other fields
  const prevSib = el.previousElementSibling;
  if (prevSib && prevSib.textContent.trim().length < 100) {
    const tag = prevSib.tagName.toLowerCase();
    // Only use simple inline elements as label sources, not containers
    if (['span', 'label', 'b', 'strong', 'em', 'i', 'small', 'p'].includes(tag)) {
      return prevSib.textContent.trim();
    }
  }
  
  return '';
}

/**
 * Find the nearest section heading above this element.
 * Looks for h1-h6, legend, or .section-title class.
 */
function findSectionHeading(el) {
  let node = el.parentElement;
  const docBody = (typeof document !== 'undefined' ? document.body : null) || el.ownerDocument?.body;
  while (node && node !== docBody) {
    // Check for heading siblings
    const headings = node.querySelectorAll('h1,h2,h3,h4,h5,h6,legend,.section-title');
    for (const h of headings) {
      // Only count if the heading is BEFORE our element in DOM order
      const POSITION_FOLLOWING = typeof Node !== 'undefined' ? Node.DOCUMENT_POSITION_FOLLOWING : 4;
      if (h.compareDocumentPosition(el) & POSITION_FOLLOWING) {
        return h.textContent.trim();
      }
    }
    
    // Check fieldset legend
    const fieldset = node.closest('fieldset');
    if (fieldset) {
      const legend = fieldset.querySelector('legend');
      if (legend) return legend.textContent.trim();
    }
    
    node = node.parentElement;
  }
  return '';
}

/**
 * Traverse open Shadow DOM roots to find form fields inside Web Components.
 * Workday and similar apps use custom elements with Shadow DOM.
 * Only works with open shadow roots (mode: 'open').
 */
function collectShadowDOMFields(root, selectors) {
  const results = [];
  const selectorStr = selectors.join(',');
  
  function walk(node) {
    if (!node) return;
    // Check if this element has an open shadow root
    if (node.shadowRoot) {
      const fields = node.shadowRoot.querySelectorAll(selectorStr);
      for (const f of fields) {
        results.push(f);
      }
      // Recurse into shadow root children
      for (const child of node.shadowRoot.children) {
        walk(child);
      }
    }
    // Recurse into light DOM children
    if (node.children) {
      for (const child of node.children) {
        walk(child);
      }
    }
  }
  
  walk(root);
  return results;
}

// ─── Exports ───────────────────────────────────────────────────────────────

if (typeof module !== 'undefined' && module.exports) {
  module.exports = { buildFieldSignatures, extractLabelText, findSectionHeading, collectShadowDOMFields };
}
