/**
 * mapper.js — Profile-to-Form Field Mapper
 * 
 * Maps profile fields to detected form fields using:
 * 1. Deterministic synonym matching
 * 2. Heuristic label similarity + type checks
 * 3. Confidence scoring (0-1)
 */

// Load synonyms — works in both extension and Node.js contexts
let SYNONYMS = {};
if (typeof require !== 'undefined') {
  try { SYNONYMS = require('./synonyms.json'); } catch (e) {}
} else if (typeof fetch !== 'undefined') {
  // Will be loaded async in extension context
}

/**
 * Set synonyms dictionary (for testing or async loading).
 */
function setSynonyms(syn) {
  SYNONYMS = syn;
}

/**
 * Resolve a dotted profile key to a value in the profile object.
 * Handles array notation like "addresses[0].city".
 */
function resolveProfileValue(profile, key) {
  const parts = key.replace(/\[(\d+)\]/g, '.$1').split('.');
  let val = profile;
  for (const part of parts) {
    if (val === undefined || val === null) return undefined;
    val = val[part];
  }
  return val;
}

/**
 * Map profile fields to form field signatures.
 * @param {Array} signatures - From buildFieldSignatures()
 * @param {Object} profile - User profile object
 * @returns {Array<{element, profileKey, value, confidence, reason}>}
 */
function mapFields(signatures, profile) {
  const results = [];
  
  for (const { element, signature } of signatures) {
    let bestMatch = null;
    let bestConfidence = 0;
    let bestReason = '';
    
    // ─── Pass 1: Deterministic synonym matching ──────────────────────
    for (const [profileKey, synonyms] of Object.entries(SYNONYMS)) {
      const conf = synonymMatch(signature, synonyms);
      if (conf > bestConfidence) {
        bestConfidence = conf;
        bestMatch = profileKey;
        bestReason = 'synonym';
      }
    }
    
    // ─── Pass 2: Heuristic matching ──────────────────────────────────
    if (bestConfidence < 0.8) {
      for (const [profileKey, synonyms] of Object.entries(SYNONYMS)) {
        const conf = heuristicMatch(signature, profileKey, synonyms);
        if (conf > bestConfidence) {
          bestConfidence = conf;
          bestMatch = profileKey;
          bestReason = 'heuristic';
        }
      }
    }
    
    // ─── Pass 3: Type-based matching ─────────────────────────────────
    if (bestConfidence < 0.5) {
      const typeMatch = typeBasedMatch(signature);
      if (typeMatch && typeMatch.confidence > bestConfidence) {
        bestConfidence = typeMatch.confidence;
        bestMatch = typeMatch.profileKey;
        bestReason = 'type-hint';
      }
    }
    
    // ─── Pass 4: Autocomplete attribute ──────────────────────────────
    if (signature.autocomplete) {
      const autoMatch = autocompleteMatch(signature.autocomplete);
      if (autoMatch && autoMatch.confidence > bestConfidence) {
        bestConfidence = autoMatch.confidence;
        bestMatch = autoMatch.profileKey;
        bestReason = 'autocomplete';
      }
    }
    
    if (bestMatch) {
      const value = resolveProfileValue(profile, bestMatch);
      if (value !== undefined && value !== null && value !== '') {
        results.push({
          element,
          profileKey: bestMatch,
          value: String(value),
          confidence: Math.round(bestConfidence * 100) / 100,
          reason: bestReason,
          signature
        });
      }
    }
  }
  
  return results;
}

/**
 * Check if any synonym matches the field's name, id, or label exactly.
 */
function synonymMatch(signature, synonyms) {
  const targets = [
    signature.name, signature.id, signature.placeholder,
    signature.ariaLabel, signature.labelText, signature.dataAutomationId
  ].filter(Boolean);
  
  for (const synonym of synonyms) {
    const syn = synonym.toLowerCase();
    for (const target of targets) {
      // Exact match
      if (target === syn) return 1.0;
      // Match with underscores/hyphens/punctuation normalized
      const normalized = target.replace(/[-_.\s?!*:;,()]/g, '');
      const synNormalized = syn.replace(/[-_.\s?!*:;,()]/g, '');
      if (normalized === synNormalized) return 0.95;
    }
  }
  return 0;
}

/**
 * Fuzzy heuristic matching using word overlap and containment.
 */
function heuristicMatch(signature, profileKey, synonyms) {
  const allText = signature.allText;
  if (!allText) return 0;
  
  // Check if any synonym words appear in the combined text
  let maxScore = 0;
  for (const synonym of synonyms) {
    const synWords = synonym.toLowerCase().split(/[-_.\s]+/).filter(w => w.length > 2);
    if (synWords.length === 0) continue;
    
    const matchedWords = synWords.filter(w => allText.includes(w));
    const score = matchedWords.length / synWords.length;
    
    if (score > maxScore) maxScore = score;
  }
  
  // Also check profile key parts (e.g., "firstName" → "first", "name")
  const keyParts = profileKey.replace(/\[\d+\]/g, '').split('.').pop()
    .replace(/([A-Z])/g, ' $1').toLowerCase().split(/\s+/).filter(w => w.length > 1);
  
  if (keyParts.length > 0) {
    const matched = keyParts.filter(w => allText.includes(w));
    // Require ALL key parts to match for multi-word keys (e.g., "first" AND "name" for firstName)
    // This prevents "last name" from matching firstName just because "name" appears
    const keyScore = matched.length === keyParts.length
      ? (matched.length / keyParts.length * 0.75)
      : (matched.length / keyParts.length * 0.3); // Heavy penalty for partial matches
    if (keyScore > maxScore) maxScore = keyScore;
  }
  
  return Math.min(maxScore, 0.85); // Cap heuristic confidence
}

/**
 * Match based on input type attribute alone.
 */
function typeBasedMatch(signature) {
  const typeMap = {
    'email': { profileKey: 'identity.email', confidence: 0.7 },
    'tel': { profileKey: 'identity.phone', confidence: 0.6 },
    'date': { profileKey: 'identity.dateOfBirth', confidence: 0.3 }, // Low — many date fields
  };
  return typeMap[signature.type] || null;
}

/**
 * Match based on HTML autocomplete attribute.
 */
function autocompleteMatch(autocomplete) {
  const map = {
    'given-name': { profileKey: 'identity.firstName', confidence: 0.95 },
    'family-name': { profileKey: 'identity.lastName', confidence: 0.95 },
    'additional-name': { profileKey: 'identity.middleName', confidence: 0.9 },
    'email': { profileKey: 'identity.email', confidence: 0.95 },
    'tel': { profileKey: 'identity.phone', confidence: 0.9 },
    'bday': { profileKey: 'identity.dateOfBirth', confidence: 0.9 },
    'street-address': { profileKey: 'addresses[0].street', confidence: 0.9 },
    'address-line1': { profileKey: 'addresses[0].street', confidence: 0.9 },
    'address-line2': { profileKey: 'addresses[0].street2', confidence: 0.9 },
    'address-level2': { profileKey: 'addresses[0].city', confidence: 0.9 },
    'address-level1': { profileKey: 'addresses[0].state', confidence: 0.9 },
    'postal-code': { profileKey: 'addresses[0].zip', confidence: 0.9 },
    'country-name': { profileKey: 'addresses[0].country', confidence: 0.9 },
    'organization': { profileKey: 'education[0].school', confidence: 0.5 },
  };
  return map[autocomplete] || null;
}

// ─── Exports ───────────────────────────────────────────────────────────────

if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    mapFields, synonymMatch, heuristicMatch, typeBasedMatch,
    autocompleteMatch, resolveProfileValue, setSynonyms
  };
}
