/**
 * content_script.js — ApplyOnce Content Script
 * 
 * Injected into the active tab when user clicks "Fill This Form".
 * Detects fields, maps profile, shows review overlay, fills on confirm.
 * NEVER clicks submit/send/pay buttons.
 */

(function() {
  'use strict';

  // Prevent double-injection
  if (window.__applyonce_injected) return;
  window.__applyonce_injected = true;

  // ─── Load core modules inline (content scripts can't use ES modules) ────
  // These will be injected by the service worker or loaded via script tags
  
  let profile = null;
  let mappings = [];

  // Listen for messages from popup/service worker
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    if (msg.action === 'fill_form') {
      profile = msg.profile;
      startFillFlow();
      sendResponse({ ok: true });
    }
    if (msg.action === 'panic') {
      removeOverlay();
      clearHighlights();
      sendResponse({ ok: true });
    }
    return true;
  });

  /**
   * Main fill flow: detect → map → show review → fill on confirm.
   */
  function startFillFlow() {
    if (!profile) return;
    
    // Step 1: Build field signatures
    const signatures = buildFieldSignaturesInline(document);
    
    // Step 2: Map profile to fields
    mappings = mapFieldsInline(signatures, profile);
    
    // Step 3: Show review overlay
    showReviewOverlay(mappings);
  }

  /**
   * Inline field signature builder (no module import in content scripts).
   */
  function buildFieldSignaturesInline(root) {
    const selectors = [
      'input[type="text"]', 'input[type="email"]', 'input[type="tel"]',
      'input[type="date"]', 'input[type="number"]', 'input[type="url"]',
      'input[type="radio"]', 'input[type="checkbox"]',
      'input:not([type])', 'textarea', 'select'
    ];
    
    const elements = root.querySelectorAll(selectors.join(','));
    const results = [];
    
    for (const el of elements) {
      if (el.type === 'hidden' || el.type === 'submit' || el.type === 'button') continue;
      if (el.disabled) continue;
      
      const sig = {
        name: (el.name || '').toLowerCase().trim(),
        id: (el.id || '').toLowerCase().trim(),
        type: (el.type || 'text').toLowerCase(),
        tagName: el.tagName.toLowerCase(),
        placeholder: (el.placeholder || '').toLowerCase().trim(),
        ariaLabel: (el.getAttribute('aria-label') || '').toLowerCase().trim(),
        labelText: extractLabel(el).toLowerCase().trim(),
        autocomplete: (el.getAttribute('autocomplete') || '').toLowerCase().trim(),
        dataAutomationId: (el.getAttribute('data-automation-id') || '').toLowerCase().trim(),
      };
      
      sig.allText = [sig.name, sig.id, sig.placeholder, sig.ariaLabel, sig.labelText, sig.autocomplete, sig.dataAutomationId]
        .filter(Boolean).join(' ').toLowerCase();
      
      results.push({ element: el, signature: sig });
    }
    return results;
  }
  
  function extractLabel(el) {
    if (el.id) {
      const label = document.querySelector(`label[for="${CSS.escape(el.id)}"]`);
      if (label) return label.textContent.trim();
    }
    const parent = el.closest('label');
    if (parent) {
      const clone = parent.cloneNode(true);
      clone.querySelectorAll('input,select,textarea').forEach(x => x.remove());
      if (clone.textContent.trim()) return clone.textContent.trim();
    }
    const prev = el.previousElementSibling;
    if (prev && prev.textContent.trim().length < 100) return prev.textContent.trim();
    const td = el.closest('td');
    if (td && td.previousElementSibling) return td.previousElementSibling.textContent.trim();
    return '';
  }

  /**
   * Inline mapper (synonym matching + heuristics).
   */
  function mapFieldsInline(signatures, prof) {
    // Inline synonym map (subset — loaded from synonyms.json at build time)
    const SYNONYMS = APPLYONCE_SYNONYMS || {};
    const results = [];
    
    for (const { element, signature } of signatures) {
      let bestKey = null, bestConf = 0, bestReason = '';
      
      for (const [profileKey, synonyms] of Object.entries(SYNONYMS)) {
        // Exact synonym match
        const targets = [signature.name, signature.id, signature.placeholder, signature.ariaLabel, signature.labelText, signature.dataAutomationId].filter(Boolean);
        for (const syn of synonyms) {
          const s = syn.toLowerCase();
          for (const t of targets) {
            const tn = t.replace(/[-_.\s]/g, '');
            const sn = s.replace(/[-_.\s]/g, '');
            if (t === s && 1.0 > bestConf) { bestConf = 1.0; bestKey = profileKey; bestReason = 'synonym'; }
            else if (tn === sn && 0.95 > bestConf) { bestConf = 0.95; bestKey = profileKey; bestReason = 'synonym'; }
          }
        }
        
        // Heuristic: word overlap
        if (bestConf < 0.8 && signature.allText) {
          for (const syn of synonyms) {
            const words = syn.toLowerCase().split(/[-_.\s]+/).filter(w => w.length > 2);
            if (words.length === 0) continue;
            const matched = words.filter(w => signature.allText.includes(w));
            const score = Math.min(matched.length / words.length * 0.85, 0.85);
            if (score > bestConf) { bestConf = score; bestKey = profileKey; bestReason = 'heuristic'; }
          }
        }
      }
      
      // Type-based fallback
      if (bestConf < 0.5) {
        if (signature.type === 'email') { bestConf = 0.7; bestKey = 'identity.email'; bestReason = 'type'; }
        else if (signature.type === 'tel') { bestConf = 0.6; bestKey = 'identity.phone'; bestReason = 'type'; }
      }
      
      // Autocomplete
      const autoMap = {
        'given-name': 'identity.firstName', 'family-name': 'identity.lastName',
        'email': 'identity.email', 'tel': 'identity.phone',
        'street-address': 'addresses[0].street', 'address-line1': 'addresses[0].street',
        'postal-code': 'addresses[0].zip', 'address-level2': 'addresses[0].city',
        'address-level1': 'addresses[0].state',
      };
      if (signature.autocomplete && autoMap[signature.autocomplete]) {
        if (0.95 > bestConf) { bestConf = 0.95; bestKey = autoMap[signature.autocomplete]; bestReason = 'autocomplete'; }
      }
      
      if (bestKey) {
        const val = resolveValue(prof, bestKey);
        if (val !== undefined && val !== null && val !== '') {
          results.push({ element, profileKey: bestKey, value: String(val), confidence: Math.round(bestConf * 100) / 100, reason: bestReason });
        }
      }
    }
    return results;
  }
  
  function resolveValue(obj, key) {
    const parts = key.replace(/\[(\d+)\]/g, '.$1').split('.');
    let val = obj;
    for (const p of parts) { if (!val) return undefined; val = val[p]; }
    return val;
  }

  /**
   * Show the review overlay.
   */
  function showReviewOverlay(mappings) {
    removeOverlay();
    
    const overlay = document.createElement('div');
    overlay.id = 'applyonce-overlay';
    overlay.style.cssText = `
      position: fixed; top: 20px; right: 20px; z-index: 2147483647;
      width: 380px; max-height: 80vh; overflow-y: auto;
      background: #1a1a2e; color: #e5e5e5; border-radius: 12px;
      font-family: -apple-system, system-ui, sans-serif; font-size: 14px;
      box-shadow: 0 8px 32px rgba(0,0,0,0.5); border: 1px solid #333;
    `;
    
    const needsReview = mappings.filter(m => m.confidence < 0.7).length;
    const highConf = mappings.length - needsReview;
    
    let html = `
      <div style="padding:16px 16px 12px; border-bottom:1px solid #333;">
        <div style="display:flex; justify-content:space-between; align-items:center;">
          <div style="font-size:16px; font-weight:600;">⚡ ApplyOnce</div>
          <button id="ao-close" style="background:none; border:none; color:#888; font-size:20px; cursor:pointer;">✕</button>
        </div>
        <div style="margin-top:8px; color:#aaa;">
          Will fill <strong style="color:#10b981;">${highConf}</strong> fields`;
    
    if (needsReview > 0) {
      html += `, <strong style="color:#f59e0b;">${needsReview}</strong> need review`;
    }
    
    html += `</div></div><div style="padding:12px 16px; max-height:50vh; overflow-y:auto;">`;
    
    // Sort: low confidence first
    const sorted = [...mappings].sort((a, b) => a.confidence - b.confidence);
    
    for (let i = 0; i < sorted.length; i++) {
      const m = sorted[i];
      const isLow = m.confidence < 0.7;
      const bgColor = isLow ? 'rgba(245,158,11,0.1)' : 'rgba(16,185,129,0.05)';
      const borderColor = isLow ? '#f59e0b' : '#333';
      const confColor = isLow ? '#f59e0b' : '#10b981';
      const label = m.profileKey.split('.').pop().replace(/([A-Z])/g, ' $1').trim();
      
      html += `
        <div style="padding:8px 10px; margin-bottom:6px; background:${bgColor}; border:1px solid ${borderColor}; border-radius:6px;">
          <div style="display:flex; justify-content:space-between;">
            <label style="display:flex; align-items:center; gap:6px; cursor:pointer;">
              <input type="checkbox" data-ao-idx="${i}" ${isLow ? '' : 'checked'} style="accent-color:#10b981;">
              <span style="font-weight:500;">${escapeHtml(label)}</span>
            </label>
            <span style="color:${confColor}; font-size:12px;">${Math.round(m.confidence * 100)}%</span>
          </div>
          <div style="margin-top:4px; color:#888; font-size:12px; overflow:hidden; text-overflow:ellipsis;">
            ${escapeHtml(m.value.substring(0, 50))}${m.value.length > 50 ? '...' : ''}
          </div>
        </div>`;
    }
    
    html += `</div>
      <div style="padding:12px 16px; border-top:1px solid #333; display:flex; gap:8px;">
        <button id="ao-fill" style="flex:1; padding:10px; background:#10b981; color:white; border:none; border-radius:8px; font-weight:600; cursor:pointer; font-size:14px;">
          ✅ Apply Fill
        </button>
        <button id="ao-panic" style="padding:10px 16px; background:#ef4444; color:white; border:none; border-radius:8px; font-weight:600; cursor:pointer; font-size:14px;">
          ✕ Cancel
        </button>
      </div>
      <div style="padding:8px 16px 12px; color:#666; font-size:11px; text-align:center;">
        ApplyOnce never clicks Submit. You stay in control.
      </div>`;
    
    overlay.innerHTML = html;
    document.body.appendChild(overlay);
    
    // Bind events
    overlay.querySelector('#ao-close').addEventListener('click', removeOverlay);
    overlay.querySelector('#ao-panic').addEventListener('click', () => {
      removeOverlay();
      clearHighlights();
    });
    overlay.querySelector('#ao-fill').addEventListener('click', () => {
      const checkboxes = overlay.querySelectorAll('input[type="checkbox"]');
      const toFill = [];
      checkboxes.forEach(cb => {
        if (cb.checked) {
          const idx = parseInt(cb.dataset.aoIdx);
          toFill.push(sorted[idx]);
        }
      });
      applyFill(toFill);
      removeOverlay();
    });
  }
  
  /**
   * Fill the selected fields.
   */
  // US state abbreviation mapping for fuzzy dropdown matching
  const STATE_MAP = {
    'alabama':'AL','alaska':'AK','arizona':'AZ','arkansas':'AR','california':'CA',
    'colorado':'CO','connecticut':'CT','delaware':'DE','florida':'FL','georgia':'GA',
    'hawaii':'HI','idaho':'ID','illinois':'IL','indiana':'IN','iowa':'IA',
    'kansas':'KS','kentucky':'KY','louisiana':'LA','maine':'ME','maryland':'MD',
    'massachusetts':'MA','michigan':'MI','minnesota':'MN','mississippi':'MS',
    'missouri':'MO','montana':'MT','nebraska':'NE','nevada':'NV','new hampshire':'NH',
    'new jersey':'NJ','new mexico':'NM','new york':'NY','north carolina':'NC',
    'north dakota':'ND','ohio':'OH','oklahoma':'OK','oregon':'OR','pennsylvania':'PA',
    'rhode island':'RI','south carolina':'SC','south dakota':'SD','tennessee':'TN',
    'texas':'TX','utah':'UT','vermont':'VT','virginia':'VA','washington':'WA',
    'west virginia':'WV','wisconsin':'WI','wyoming':'WY',
    'district of columbia':'DC'
  };
  const STATE_ABBR_TO_FULL = {};
  for (const [full, abbr] of Object.entries(STATE_MAP)) {
    STATE_ABBR_TO_FULL[abbr.toLowerCase()] = full;
  }

  function fuzzySelectMatch(el, value) {
    const options = el.querySelectorAll('option');
    const val = value.toLowerCase().trim();
    
    // Pass 1: exact match on value or text
    for (const opt of options) {
      if (opt.value.toLowerCase().trim() === val || opt.textContent.trim().toLowerCase() === val) {
        el.value = opt.value; return true;
      }
    }
    // Pass 2: abbreviation ↔ full name (states)
    const altVal = STATE_MAP[val] || STATE_ABBR_TO_FULL[val];
    if (altVal) {
      for (const opt of options) {
        const ov = opt.value.toLowerCase().trim();
        const ot = opt.textContent.trim().toLowerCase();
        if (ov === altVal.toLowerCase() || ot === altVal.toLowerCase() ||
            ov === val || ot === val) {
          el.value = opt.value; return true;
        }
      }
    }
    // Pass 3: substring/contains match (e.g., "Calif." matches "California")
    for (const opt of options) {
      const ot = opt.textContent.trim().toLowerCase();
      if (ot.includes(val) || val.includes(ot)) {
        if (ot.length > 1 && val.length > 1) { el.value = opt.value; return true; }
      }
    }
    // Pass 4: case-insensitive startsWith
    for (const opt of options) {
      const ot = opt.textContent.trim().toLowerCase();
      if (ot.startsWith(val) || val.startsWith(ot)) {
        if (ot.length > 1) { el.value = opt.value; return true; }
      }
    }
    return false;
  }

  function applyFill(items) {
    for (const item of items) {
      const el = item.element;
      if (!el || !el.isConnected) continue;
      
      if (el.tagName === 'SELECT') {
        // Fuzzy dropdown matching with state abbreviation support
        if (!fuzzySelectMatch(el, item.value)) {
          el.value = item.value; // Last resort: direct set
        }
      } else if (el.type === 'radio') {
        // Radio: check if this option's value or label matches
        const val = item.value.toLowerCase().trim();
        const label = extractLabel(el).toLowerCase().trim();
        const elVal = (el.value || '').toLowerCase().trim();
        // Match on: "yes"/"no", "true"/"false", exact value, or label text
        const yesValues = ['yes', 'true', '1', 'y'];
        const noValues = ['no', 'false', '0', 'n'];
        if (yesValues.includes(val) && yesValues.includes(elVal)) { el.checked = true; }
        else if (noValues.includes(val) && noValues.includes(elVal)) { el.checked = true; }
        else if (elVal === val || label.includes(val) || val.includes(label)) { el.checked = true; }
      } else if (el.type === 'checkbox') {
        // Checkbox: check if value indicates "yes"/"true"
        const val = item.value.toLowerCase().trim();
        const yesValues = ['yes', 'true', '1', 'y', 'checked'];
        el.checked = yesValues.includes(val);
      } else {
        el.value = item.value;
      }
      
      // Dispatch events to trigger form validation
      el.dispatchEvent(new Event('input', { bubbles: true }));
      el.dispatchEvent(new Event('change', { bubbles: true }));
      el.dispatchEvent(new Event('blur', { bubbles: true }));
      
      // Highlight based on confidence
      if (item.confidence < 0.7) {
        el.style.outline = '2px solid #f59e0b';
        el.style.outlineOffset = '1px';
        el.dataset.aoReview = 'true';
      } else {
        el.style.outline = '2px solid #10b981';
        el.style.outlineOffset = '1px';
      }
      
      // Remove highlight after 5 seconds
      setTimeout(() => {
        if (el.dataset.aoReview !== 'true') {
          el.style.outline = '';
          el.style.outlineOffset = '';
        }
      }, 5000);
    }
  }
  
  function removeOverlay() {
    const overlay = document.getElementById('applyonce-overlay');
    if (overlay) overlay.remove();
  }
  
  function clearHighlights() {
    document.querySelectorAll('[data-ao-review]').forEach(el => {
      el.style.outline = '';
      el.style.outlineOffset = '';
      delete el.dataset.aoReview;
    });
  }
  
  function escapeHtml(str) {
    const div = document.createElement('div');
    div.textContent = str;
    return div.innerHTML;
  }
  
  // Synonyms will be injected by service worker before this script runs
  const APPLYONCE_SYNONYMS = window.__applyonce_synonyms || {};
  
})();
